<?php

namespace App\Modules\ModuleManager\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\File;
use Illuminate\Http\Request;

class ModuleManagerController extends Controller
{
    /**
     * 📦 Get all modules from Core and Modules directories
     */
    public function test()
    {
        return "??";
    }

    public function index()
    {
        $modules = $this->loadModules();
        return response()->json([
            'status' => 'success',
            'total' => count($modules),
            'modules' => $modules
        ]);
    }

    /**
     * 🔍 Get specific module details
     */
    public function show($slug)
    {
        $modules = $this->loadModules();
        $module = collect($modules)->firstWhere('slug', $slug);

        if (!$module) {
            return response()->json(['error' => 'Module not found'], 404);
        }

        return response()->json(['module' => $module]);
    }

    /**
     * ✅ Enable a module (updates module.json flag)
     */
    public function enable($slug)
    {
        return $this->toggleModule($slug, true);
    }

    /**
     * 🚫 Disable a module (updates module.json flag)
     */
    public function disable($slug)
    {
        return $this->toggleModule($slug, false);
    }

    /**
     * ♻️ Reload or refresh module cache
     */
    public function refresh()
    {
        $modules = $this->loadModules(true); // force reload
        return response()->json([
            'message' => 'Module cache refreshed',
            'modules' => $modules
        ]);
    }

    // ---------------------------------------------------------------
    // 🔧 Internal helper functions
    // ---------------------------------------------------------------

    private function loadModules($refresh = false)
    {
        $basePaths = [base_path('app/Core'), base_path('app/Modules')];
        $modules = [];

        foreach ($basePaths as $path) {
            if (!File::isDirectory($path)) continue;

            $folders = File::directories($path);
            foreach ($folders as $folder) {
                $manifest = $folder . '/module.json';
                if (File::exists($manifest)) {
                    $json = json_decode(File::get($manifest), true);
                    if ($json) $modules[] = $json;
                }
            }
        }

        return $modules;
    }

    private function toggleModule($slug, $state)
    {
        $paths = [base_path('app/Core'), base_path('app/Modules')];
        foreach ($paths as $path) {
            $folders = File::directories($path);
            foreach ($folders as $folder) {
                $manifest = $folder . '/module.json';
                if (File::exists($manifest)) {
                    $json = json_decode(File::get($manifest), true);
                    if (isset($json['slug']) && $json['slug'] === $slug) {
                        $json['enabled'] = $state;
                        File::put($manifest, json_encode($json, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
                        return response()->json([
                            'message' => "Module '{$slug}' has been " . ($state ? 'enabled' : 'disabled'),
                            'module' => $json
                        ]);
                    }
                }
            }
        }

        return response()->json(['error' => 'Module not found'], 404);
    }
}
