<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    protected string $p = 'seed_';

    public function up(): void
    {
        $p = $this->p;

        /*-------------------------------------------------------
        | 1️⃣ Seed Types (Wheat, Cotton, Hybrid Corn etc.)
        -------------------------------------------------------*/
        Schema::create($p.'types', function (Blueprint $table) {
            $table->id();
            $table->string('name')->unique();
            $table->text('description')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 2️⃣ Growers (Users)
        -------------------------------------------------------*/
        Schema::create($p.'growers', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('user_id'); // removed FK
            $table->string('farm_location')->nullable();
            $table->string('farm_size_acres')->nullable();
            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 3️⃣ Seed Staff (Team Leaders, Lab Techs etc.)
        -------------------------------------------------------*/
        Schema::create($p.'staff', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('user_id'); // removed FK

            $table->enum('role', [
                'team_leader',
                'field_worker',
                'lab_technician',
                'processing_supervisor',
                'loader'
            ]);

            $table->boolean('is_active')->default(true);
            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 4️⃣ Seed Batches
        -------------------------------------------------------*/
        Schema::create($p.'batches', function (Blueprint $table) use ($p) {
            $table->id();
            $table->uuid('uuid')->unique();

            $table->unsignedBigInteger('seed_type_id'); // removed FK
            $table->unsignedBigInteger('grower_id'); // removed FK

            $table->string('batch_no')->unique();
            $table->decimal('weight_kg', 10, 2)->nullable();

            $table->enum('status', ['received', 'processing', 'completed', 'failed'])
                ->default('received');

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 5️⃣ Processing Stages (Dynamic)
        -------------------------------------------------------*/
        Schema::create($p.'processing_stages', function (Blueprint $table) {
            $table->id();
            $table->string('name'); // Cleaning, Drying, Grading, Treating etc.
            $table->integer('sort_order')->default(0);
            $table->boolean('is_active')->default(true);
            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 6️⃣ Processing Jobs
        -------------------------------------------------------*/
        Schema::create($p.'processing_jobs', function (Blueprint $table) use ($p) {
            $table->id();
            $table->uuid('uuid')->unique();

            $table->unsignedBigInteger('batch_id'); // removed FK
            $table->unsignedBigInteger('stage_id'); // removed FK
            $table->unsignedBigInteger('assigned_to')->nullable(); // removed FK

            $table->enum('status', ['pending','running','completed'])->default('pending');

            $table->timestamp('started_at')->nullable();
            $table->timestamp('completed_at')->nullable();

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 7️⃣ Field Visits (GPS)
        -------------------------------------------------------*/
        Schema::create($p.'field_visits', function (Blueprint $table) use ($p) {
            $table->id();
            $table->uuid('uuid')->unique();

            $table->unsignedBigInteger('batch_id')->nullable(); // removed FK
            $table->unsignedBigInteger('grower_id'); // removed FK

            $table->decimal('latitude', 10, 7)->nullable();
            $table->decimal('longitude', 10, 7)->nullable();
            $table->string('location_text')->nullable();
            $table->date('visited_on');

            $table->text('notes')->nullable();
            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 8️⃣ Field Visit Members (Team visiting grower)
        -------------------------------------------------------*/
        Schema::create($p.'field_visit_members', function (Blueprint $table) use ($p) {
            $table->id();

            $table->unsignedBigInteger('visit_id'); // removed FK
            $table->unsignedBigInteger('staff_id'); // removed FK

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 9️⃣ Field Visit Results
        -------------------------------------------------------*/
        Schema::create($p.'field_visit_results', function (Blueprint $table) use ($p) {
            $table->id();

            $table->unsignedBigInteger('visit_id'); // removed FK

            $table->json('inspection_data')->nullable(); // moisture, disease, growth stage
            $table->text('remarks')->nullable();
            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 🔟 Team Assignments (Daily task teams)
        -------------------------------------------------------*/
        Schema::create($p.'team_assignments', function (Blueprint $table) use ($p) {
            $table->id();
            $table->unsignedBigInteger('team_leader_id'); // removed FK

            $table->date('assignment_date');
            $table->string('area')->nullable();
            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 1️⃣1️⃣ Team Assignment Members
        -------------------------------------------------------*/
        Schema::create($p.'team_assignment_members', function (Blueprint $table) use ($p) {
            $table->id();

            $table->unsignedBigInteger('assignment_id'); // removed FK
            $table->unsignedBigInteger('staff_id'); // removed FK

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 1️⃣2️⃣ Lab Test Types
        -------------------------------------------------------*/
        Schema::create($p.'lab_test_types', function (Blueprint $table) {
            $table->id();
            $table->string('name'); // Germination, Purity, Moisture
            $table->boolean('is_active')->default(true);
            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 1️⃣3️⃣ Lab Tests
        -------------------------------------------------------*/
        Schema::create($p.'lab_tests', function (Blueprint $table) use ($p) {
            $table->id();
            $table->uuid('uuid')->unique();

            $table->unsignedBigInteger('batch_id'); // removed FK
            $table->unsignedBigInteger('test_type_id'); // removed FK
            $table->unsignedBigInteger('technician_id')->nullable(); // removed FK

            $table->date('tested_on')->nullable();
            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 1️⃣4️⃣ Lab Test Results
        -------------------------------------------------------*/
        Schema::create($p.'lab_test_results', function (Blueprint $table) use ($p) {
            $table->id();

            $table->unsignedBigInteger('test_id'); // removed FK
            $table->json('results')->nullable(); // moisture %, germination %, etc.
            $table->text('remarks')->nullable();
            $table->timestampsTz();
        });
    }

    public function down(): void
    {
        $p = $this->p;

        Schema::dropIfExists($p.'lab_test_results');
        Schema::dropIfExists($p.'lab_tests');
        Schema::dropIfExists($p.'lab_test_types');

        Schema::dropIfExists($p.'team_assignment_members');
        Schema::dropIfExists($p.'team_assignments');

        Schema::dropIfExists($p.'field_visit_results');
        Schema::dropIfExists($p.'field_visit_members');
        Schema::dropIfExists($p.'field_visits');

        Schema::dropIfExists($p.'processing_jobs');
        Schema::dropIfExists($p.'processing_stages');

        Schema::dropIfExists($p.'batches');
        Schema::dropIfExists($p.'staff');
        Schema::dropIfExists($p.'growers');
        Schema::dropIfExists($p.'types');
    }
};
